<?php

namespace NGD_CCM;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use NGD_CCM\Front\Front;
use NGD_CCM\Admin\Admin;
use NGD_CCM\Functions\Functions;

class App {

	public function __construct( $run_hooks = false ) {
		if ( $run_hooks ) {
			$this->run_hooks();
			new Front( $run_hooks );
			if ( is_admin() ) {
				new Admin( $run_hooks );
			}
		}
	}

	public function run_hooks(): void {
		add_action( 'init', [ $this, 'load_textdomain' ] );
	}

	public function load_textdomain(): void {
		load_plugin_textdomain( 'cookiewatch-ccm', false, NGD_CCM_DIR . '/languages' );
	}

	public static function get_domain(): string {

		return trim( str_replace( [ 'http://', 'https://' ], '', get_site_url() ), '/' );
	}

	public static function get_domain_data(): array {

		$response_data = [
			'domain_id'      => false,
			'domain_api_key' => false,
			'status'         => false,
			'message'        => esc_html__( 'Something went wrong...', 'cookiewatch-ccm' )
		];
		$domain        = self::get_domain();
		$api_key       = get_option( 'cookiewatch_ccm_api_key', '' );
		$url           = 'https://api.cookiewatch.de/public/DomainCheck?domain=' . $domain . '&apikey=' . $api_key;

		$response      = wp_remote_get( $url );
		$response_code = wp_remote_retrieve_response_code( $response );

		if ( is_array( $response ) && ! is_wp_error( $response ) ) {
			if ( $response_code == 200 ) {
				$body = $response['body']; // use the content
				if ( self::is_json( $body ) ) {
					$body = json_decode( $body, 1 );
					if ( ! empty( $body['domainId'] ) ) {
						$response_data['domain_id'] = esc_attr( $body['domainId'] );
					}
					if ( ! empty( $body['apiKey'] ) ) {
						$response_data['domain_api_key'] = esc_attr( $body['apiKey'] );
					}
					if ( $response_data['domain_id'] && $response_data['domain_api_key'] ) {
						$response_data['status']  = true;
						$response_data['message'] = true;
					}
				}
			} else if ( $response_code == 404 ) {
				$response_data['message'] = esc_html__( 'Unknown Domain Name. Please register your domain on cookiewatch.de', 'cookiewatch-ccm' );
			}
		} else if ( is_wp_error( $response ) ) {
			$response_data['message'] = $response->get_error_message();
		}

		return $response_data;
	}

	public static function is_json( $string ): bool {
		if ( is_numeric( $string ) ) {
			return false;
		}
		if ( is_null( $string ) ) {
			return false;
		}
		if ( is_bool( $string ) ) {
			return false;
		}
		if ( ! is_string( $string ) ) {
			return false;
		}
		if ( $string == "" || $string == " " ) {
			return false;
		}
		@json_decode( $string );

		return ( json_last_error() == JSON_ERROR_NONE );
	}

	public static function get_script_to_add(): bool|string {
		$code = get_option( 'cookiewatch_ccm_script', '' );
		if ( $code ) {
			return $code;
		}

		return self::get_default_script_to_add();
	}

	public static function get_default_script_to_add(): bool|string {
		ob_start();
		$domain_id      = get_option( 'cookiewatch_ccm_domain_id', '' );
		$domain_api_key = get_option( 'cookiewatch_ccm_domain_api_key', '' );
		$src            = "https://cookiewatch.de/ccm/public/app.js?apiKey=" . $domain_api_key . "&domain=" . $domain_id;
		?>
        <script src="<?php echo $src; ?>" referrerpolicy="origin"></script>
		<?php
		return ob_get_clean();
	}

}