<?php

namespace NGD_CCM\Admin;

use NGD_CCM\App;
use NGD_CCM\Functions\Functions;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Admin {

	public function __construct( $run_hooks = false ) {
		if ( $run_hooks ) {
			$this->run_hooks();
		}
	}

	public function run_hooks(): void {
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue' ] );
		add_action( 'admin_menu', [ $this, 'menus' ] );
		add_action( 'admin_init', [ $this, 'register_settings' ] );
		add_action( 'admin_init', [ $this, 'before_saving_options' ] );
		add_filter( 'plugin_action_links_cookiewatch-ccm/cookiewatch-ccm.php', [ $this, 'settings_link' ] );
	}

	public function enqueue(): void {
		wp_enqueue_style( 'ngd_admin_styles', NGD_CCM_URL . '/assets/css/admin.css' );
	}

	public function menus(): void {
		add_menu_page(
			esc_html__( 'Cookiewatch CCM', 'cookiewatch_ccm' ),
			esc_html__( 'Cookiewatch CCM', 'cookiewatch_ccm' ),
			'manage_options',
			'cookiewatch_ccm',
			[ $this, 'menu_page_main' ],
			'data:image/svg+xml;base64,' . base64_encode( '<svg xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24"><path fill="currentColor" d="M12 2c.714 0 1.419.075 2.106.222a.75.75 0 0 1 .374 1.263a2.501 2.501 0 0 0 1.206 4.201a.75.75 0 0 1 .577.811a2.5 2.5 0 0 0 4.36 1.908a.75.75 0 0 1 1.307.409c.047.39.07.787.07 1.186c0 5.523-4.477 10-10 10S2 17.523 2 12S6.477 2 12 2Zm3 14a1 1 0 1 0 0 2a1 1 0 0 0 0-2Zm-7-1a1 1 0 1 0 0 2a1 1 0 0 0 0-2Zm4-4a1 1 0 1 0 0 2a1 1 0 0 0 0-2ZM7 8a1 1 0 1 0 0 2a1 1 0 0 0 0-2Z"/></svg>' ),
			30
		);
	}

	function settings_link( array $links ): array {
		$url           = get_admin_url() . "options-general.php?page=cookiewatch_ccm";
		$settings_link = '<a href="' . $url . '">' . __( 'Settings', 'cookiewatch-ccm' ) . '</a>';
		$links[]       = $settings_link;

		return $links;
	}

	public function menu_page_main(): void {
		ob_start();
		?>
        <div class="wrap cookiewatch_ccm_admin_page">
            <h1></h1>
            <div class="cookiewatch_ccm_wrap">
                <h1 class="page-title"><?php esc_html_e( 'Cookiewatch Cookie-Consent Manager', 'cookiewatch-ccm' ); ?></h1>

				<?php
				$domain_data = App::get_domain_data();
				?>
				<?php if ( $domain_data['status'] ) {
					update_option( 'cookiewatch_ccm_domain_id', $domain_data['domain_id'] );
					update_option( 'cookiewatch_ccm_domain_api_key', $domain_data['domain_api_key'] );
					?>
				<?php } else { ?>
                    <div class="cookiewatch_admin_notice cookiewatch_admin_notice_error">
                        <p>
							<?php echo $domain_data['message']; ?>
                        </p>
                    </div>
				<?php } ?>
                <form method="post" action="options.php" class="cookiewatch_ccm_settings_form">
					<?php settings_fields( 'cookiewatch_ccm' ); ?>
                    <input type="hidden" name="cookiewatch_ccm_enable_script">
                    <div class="cookiewatch_ccm_form_fields_section">
                        <div class="cookiewatch_ccm_form_fields_section_content">
                            <div class="cookiewatch_ccm_form_field" style="display: none;">
								<?php
								$field_label = 'Cookiewatch API Key';
								$field_name  = 'cookiewatch_ccm_api_key';
								$field_value = get_option( $field_name, '' );
								?>
                                <label class="cookiewatch_ccm_form_label"
                                       for="<?php echo $field_name; ?>"><?php echo $field_label; ?></label>
                                <input name="<?php echo $field_name; ?>" id="<?php echo $field_name; ?>"
                                       type="text" value="<?php echo $field_value; ?>">
                            </div>
                            <div class="cookiewatch_ccm_form_field">
								<?php
								$field_label = 'Domain Name';
								$field_name  = 'cookiewatch_ccm_domain_name';
								$field_value = App::get_domain();
								?>
                                <label class="cookiewatch_ccm_form_label"
                                       for="<?php echo $field_name; ?>"><?php echo $field_label; ?></label>
                                <input id="<?php echo $field_name; ?>" readonly
                                       type="text" value="<?php echo $field_value; ?>">
                            </div>
                            <div class="cookiewatch_ccm_form_field">
								<?php
								$field_label = 'Script to add';
								$field_name  = 'cookiewatch_ccm_script';
								$field_value = get_option( $field_name, App::get_script_to_add() );
								?>
                                <label class="cookiewatch_ccm_form_label"
                                       for="<?php echo $field_name; ?>"><?php echo $field_label; ?></label>
                                <textarea
                                        rows="5"
                                        id="<?php echo $field_name; ?>"
                                        name="<?php echo $field_name; ?>"
                                ><?php echo trim( esc_html( $field_value ) ); ?></textarea>
                            </div>

							<?php if ( $domain_data['status'] ) {
								$field_value = get_option( 'cookiewatch_ccm_enable_script', false );
								//echo '<pre>' . print_r($field_value, 1) . '</pre>';
								$checked = empty( $field_value ) ? '' : 'checked';
								update_option( 'cookiewatch_ccm_domain_id', $domain_data['domain_id'] );
								update_option( 'cookiewatch_ccm_domain_api_key', $domain_data['domain_api_key'] );

								$default_script  = App::get_default_script_to_add();
								$added_script    = get_option( 'cookiewatch_ccm_script', App::get_script_to_add() );
								$script_modified = trim( $default_script ) !== trim( $added_script );

								if ( $script_modified ) {
									$message_modified = esc_html__( 'Notice! You have made changes to the default script. You can enable your modified script or reset the default script by clicking on the button below', 'cookiewatch-ccm' );
								}
								if ( $checked ) {
									$message_success = esc_html__( 'Success! You have successfully activated the cookie consent notification on your site!', 'cookiewatch-ccm' );
								} else {
									$message_success = esc_html__( 'Success! You can activate cookie consent notification by checking the box below!', 'cookiewatch-ccm' );
								}

								//if ($checked)
								?>
								<?php if ( $script_modified ) { ?>
                                    <div class="cookiewatch_admin_notice cookiewatch_admin_notice_warning">
                                        <p><?php echo $message_modified; ?></p>
                                    </div>
								<?php } ?>
								<?php if ( ! empty( $added_script ) ) { ?>
                                    <div class="cookiewatch_admin_notice cookiewatch_admin_notice_success">
                                        <p><?php echo $message_success; ?></p>
                                    </div>
								<?php } ?>
                                <div class="cookiewatch_ccm_form_field cookiewatch_ccm_form_field_checkboxes">
                                    <div class="checkboxes_container">
                                        <label>
                                            <input <?php echo $checked ?>
                                                    type="checkbox"
                                                    name='cookiewatch_ccm_enable_script'
                                                    value="on">
											<?php esc_html_e( 'Enable Integration', 'cookiewatch-ccm' ); ?>
                                        </label>
                                    </div>
                                </div>
							<?php } ?>
                        </div>
                    </div>
                    <div class="cookiewatch_ccm_form_field cookiewatch_ccm_form_submit">
                        <button class="button button-primary"
                                name="submit_action"
                                value="save"
                                type="submit">
							<?php esc_html_e( 'Save Config Options', 'cookiewatch-ccm' ); ?></button>
                        <button class="button button-primary"
                                name="submit_action"
                                value="reset_script"
                                type="submit"
                                onclick="return confirm('Are you sure you want to reset the script?')"
                        >
							<?php esc_html_e( 'Reset Script', 'cookiewatch-ccm' ); ?></button>
                        <a href="https://cookiewatch.de/ccm/public/" target="_blank" class="button button-secondary"><?php esc_html_e('Open cookiewatch configuration', 'cookiewatch-ccm'); ?><span
                                    class="dashicons dashicons-external" style="vertical-align: text-bottom;"></span></a>
                    </div>
                </form>
            </div>
            <div class="cookiewatch_ccm_wrap">
                <h1 class="page-title"><?php esc_html_e( 'WordPress Setup Guidelines', 'cookiewatch-ccm' ); ?></h1>
                <p><?php _e( 'This plugin let´s you easily add your cookiewatch.de consent-manager Code-Snippet to your WordPress website.', 'cookiewatch-ccm' ); ?></p>
                <p><?php _e( 'Simply check that your WordPress Domain below matches your Domain registered on cookiewatch.de and press the setup button to enable your Consent-Manager!', 'cookiewatch-ccm' ); ?></p>
				<?php
				$link = "<a target='_blank' href='https://docs.cookiewatch.de/'>https://docs.cookiewatch.de</a>";
				?>
                <p><?php
					/* translators: %s: "<a target='_blank' href='https://docs.cookiewatch.de/Basics/Grundfunktionen/themes'>https://docs.cookiewatch.de</a>" */
					echo sprintf( __( 'If you haven´t done so already, visit %s to configure your Consent-Banner to your liking.', 'cookiewatch-ccm' ), $link ); ?></p>
				<?php
				$link  = "<a target='_blank' href='https://docs.cookiewatch.de/setup/wordpress'>https://docs.cookiewatch.de</a>";
				$email = "<a href='mailto:support@cookiewatch.de'>support@cookiewatch.de</a>";
				?>
                <p>
					<?php
					/* translators: %1$s: "<a target='_blank' href='https://docs.cookiewatch.de/setup/wordpress'>https://docs.cookiewatch.de</a>" */
					/* translators: %2$s: "<a href='mailto:support@cookiewatch.de'>support@cookiewatch.de</a>" */
					echo sprintf( __( 'Any questions? Visit our documentation at %1$s or contact us via mail at %2$s', 'cookiewatch-ccm' ), $link, $email );
					?>
                </p>
            </div>
            <div class="cookiewatch_ccm_wrap">
                <h1 class="page-title"><?php esc_html_e( 'Cookie Explanation Table', 'cookiewatch-ccm' ); ?></h1>
				<?php
				$code = "<code>[cookiewatch_ccm_table]</code>";
				?>
                <p><?php
	                /* translators: %s: $code = "<code>[cookiewatch_ccm_table]</code>" */
                    echo sprintf( __( 'You can add the shortcode %s to your GDPR privacy statement to always display the currently used cookies.', 'cookiewatch-ccm' ), $code ); ?></p>
				<?php
				$link = "<a target='_blank' href='https://docs.cookiewatch.de/Basics/Rechtliche-Anforderungen/Cookieerkl%C3%A4rungstabelle'>https://docs.cookiewatch.de/Basics/Rechtliche-Anforderungen/Cookieerkl%C3%A4rungstabelle</a>";
				?>
                <p><?php
	                /* translators: %s: $code = "<a target='_blank' href='https://docs.cookiewatch.de/Basics/Rechtliche-Anforderungen/Cookieerkl%C3%A4rungstabelle'>https://docs.cookiewatch.de/Basics/Rechtliche-Anforderungen/Cookieerkl%C3%A4rungstabelle</a>" */
                    echo sprintf( __( 'More information is available here %s', 'cookiewatch-ccm' ), $link ); ?></p>
            </div>
        </div>
		<?php


		echo ob_get_clean();
	}

	public function register_settings(): void {

		register_setting( 'cookiewatch_ccm', 'cookiewatch_ccm_api_key' );
		register_setting( 'cookiewatch_ccm', 'cookiewatch_ccm_domain_id' );
		register_setting( 'cookiewatch_ccm', 'cookiewatch_ccm_domain_api_key' );
		register_setting( 'cookiewatch_ccm', 'cookiewatch_ccm_enable_script' );
		register_setting( 'cookiewatch_ccm', 'cookiewatch_ccm_script' );
	}

	public function before_saving_options(): void {
		if ( ! empty( $_REQUEST['submit_action'] ) && $_REQUEST['submit_action'] == 'reset_script' ) {
			$_POST['cookiewatch_ccm_script'] = App::get_default_script_to_add();
		}
	}


}